//shading:         phong
//lighting model:  blinn
//lighting type:   point

// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;
float4x4 tWVP: WORLDVIEWPROJECTION;
float4x4 tWIT: WORLDINVERSETRANSPOSE;
float4x4 tWI: WORLDINVERSE;
float4x4 tWT: WORLDTRANSPOSE;
float4x4 tVI: VIEWINVERSE;
float4x4 tVP: VIEWPROJECTION;

//light properties
float3 lPos <string uiname="Light Position";> = {0, 5, -2};       //light position in world space
float4 lAmb  : COLOR <String uiname="Ambient Color";> = {0, 0, 0, 1};
float4 lDiff : COLOR <String uiname="Diffuse Color";> = {0.7, 0.7, 0.7, 1};
float4 lSpec : COLOR <String uiname="Specular Color";> = {0.8, 0.8, 0.8, 1};
float lPower <String uiname="Power"; float uimin=0.0;> = 10.0;    //shininess of specular highlight

float BumpFactor <string uiname="Bump Factor";> = 1;

//texture
texture Tex <string uiname="Texture";>;
sampler Samp = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (Tex);          //apply a texture to the sampler
    MipFilter = LINEAR;         //set the sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

texture NormalMap <string uiname="Normal Map";>;
sampler SampNormal = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (NormalMap);          //apply a texture to the sampler
    MipFilter = LINEAR;         //set the sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------

struct vs2ps {
    float4 WVP: POSITION;
    float4 TexCd: TEXCOORD0;
    float3 LightDirW: TEXCOORD1;
    float3 NormW: TEXCOORD2;
    float3 ViewDirW: TEXCOORD3;
    float3 TangentW: TEXCOORD4;
    float3 BinormalW: TEXCOORD5;
};

/*********** vertex shader ******/

vs2ps VS(
      float4 PosO: POSITION,
      float4 NormO: NORMAL,
      float4 TexCd: TEXCOORD0,
      float4 TangentO: TANGENT,
      float4 BinormalO: BINORMAL)
{
    //inititalize all fields of Output struct with 0
    vs2ps Out = (vs2ps)0;
    
    Out.NormW = normalize(mul(NormO, tWIT));
    Out.TangentW = normalize(mul(TangentO, tWIT));
    Out.BinormalW = normalize(mul(BinormalO, tWIT));
    float4 posW = mul(PosO, tW);

    Out.LightDirW = normalize(lPos - posW);
    Out.TexCd = mul(TexCd, tTex);
    float4 posV = mul(posW, tV);
    Out.ViewDirW =  normalize(tVI[3].xyz - posW);
    Out.WVP = mul(posW, tVP);
    return Out;
}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------

float4 PS(vs2ps In): COLOR
{
    //read the bump information for this pixel
    float3 bump = BumpFactor * (tex2D(SampNormal, In.TexCd) - 0.5);

    //bump the normal
    float3 nBumped = In.NormW + (bump.x * In.TangentW + bump.y * In.BinormalW);
    nBumped = normalize(nBumped);

    //halfvector
    float3 H = normalize(In.ViewDirW + In.LightDirW);
    
    //compute blinn lighting
    float3 shades = lit(dot( nBumped, In.LightDirW), dot(H, nBumped), lPower);
    
    float4 diff = lDiff * shades.y;
    diff.a = 1;
    float4 spec = lSpec * shades.z;
    spec.a = 1;
    
    float4 col = tex2D(Samp, In.TexCd) * (lAmb + diff) + spec;
    return col;
}

technique TPhonPointNormalMap
{
     pass P0
     {	
        TexCoordIndex[0] = 0;	
        TexCoordIndex[1] = 1;
        TexCoordIndex[2] = 2;
        TexCoordIndex[3] = 3;
        TexCoordIndex[4] = 4;
        TexCoordIndex[5] = 5;
	VertexShader = compile vs_1_1 VS();
	PixelShader = compile ps_2_0 PS();
     }
}
